## ----knitr_setup, include = FALSE, echo=FALSE---------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----load_packages, message=FALSE, warning=FALSE------------------------------
library(aquacultuR)
library(magrittr)
library(dplyr)
library(tidyr)
library(lubridate)

## ----options------------------------------------------------------------------
oldopts <- options()
options(digits = 3)

## ----feedcomp-----------------------------------------------------------------
feedcomp

## ----feedcomp_long------------------------------------------------------------
feedcomp %>% 
  select(-diet) %>% 
  pivot_longer(
    everything(),
    names_to = "parameter",
    values_to = "value"
  )

## -----------------------------------------------------------------------------
df <- samplings %>%
  mutate(
    timepoint = case_when(
      date == ymd("2023-03-16") ~ "bw_beginning",
      date == ymd("2023-04-14") ~ "bw_end",
      .default = NA
  )) %>%
  group_by(tank, timepoint, sample_type) %>%
  summarise(mean_weight = mean(fish_weight)) %>% 
  select(-sample_type, -starts_with("sd")) %>%
  pivot_wider(
    names_from = timepoint,
    values_from = mean_weight
  ) %>% 
  print()

## -----------------------------------------------------------------------------
head(feed_intake)

## -----------------------------------------------------------------------------
head(bodycomp)

## ----feed_intake--------------------------------------------------------------
df <- feed_intake %>% 
  filter(date == max(date)) %>% 
  select(tank, cumulative_feed_intake) %>% 
  right_join(df, join_by(tank)) %>% 
  print()

## ----calculate_fcr_fce--------------------------------------------------------
feed_conversion <- df %>%
  group_by(tank) %>%
  summarise(
    feed_conversion_ratio = fcr(
      ibw = bw_beginning,
      fbw = bw_end,
      feed = cumulative_feed_intake,
      dm = 0.93
    ),
    feed_conversion_efficiency = fce(
      ibw = bw_beginning,
      fbw = bw_end,
      feed = cumulative_feed_intake,
      dm = 0.93
    )
  ) %>%
  print()

## ----calculate_ner------------------------------------------------------------
df %>% 
  bind_cols(feedcomp %>% 
              select(dry_matter, crude_protein)) %>% 
  mutate(
    protein_efficiency_ratio = ner(ibw = bw_beginning, 
                                   fbw = bw_end, 
                                   fi = cumulative_feed_intake, 
                                   nut_f = crude_protein, 
                                   dm = dry_matter)
  ) %>% 
  relocate(tank, protein_efficiency_ratio) # change column order

## ----summarise_bodycomp-------------------------------------------------------
summary(bodycomp)

## ----assign_nr_time-----------------------------------------------------------
df2 <- bodycomp %>% 
    mutate(
      timepoint = case_when(
        date == ymd("2023-03-16") ~ "beginning", 
        date == ymd("2023-04-14") ~ "end",
        .default = NA
        )
      )

## ----prepare_nr_init_data-----------------------------------------------------
initial <- df2 %>% 
  group_by(timepoint) %>% 
  summarise(tissue_protein_start = mean(protein)) %>%
  ungroup() %>% 
  filter(timepoint == "beginning") %>% 
  select(tissue_protein_start) %>% print()

## ----join_nr_data-------------------------------------------------------------
nutrient_retention <- df2 %>%
  filter(timepoint != "beginning") %>% 
  rename(tissue_protein = "protein") %>% 
  select(-timepoint) %>% 
  bind_cols(initial) %>% 
  right_join(df, join_by(tank)) %>% 
  bind_cols(feedcomp %>% select(dry_matter, crude_protein))

## ----calculate_nr-------------------------------------------------------------
nutrient_retention %>% 
  mutate(
    protein_retention = nr(ibw = bw_beginning, 
                           fbw = bw_end, 
                           ibn = tissue_protein_start,
                           fbn = tissue_protein, 
                           fi = cumulative_feed_intake,
                           nut_f = crude_protein
                           )
    ) %>% 
  select(treatment, protein_retention)

## ----show_session_info--------------------------------------------------------
sessionInfo()

## ----reset_options------------------------------------------------------------
options(oldopts)

